#!/usr/bin/env node

/**
 * Generate `config/posthog.json` from environment variables.
 *
 * Required environment variables:
 *   - POSTHOG_PROJECT_API_KEY
 *
 * Optional environment variables:
 *   - POSTHOG_API_HOST (defaults to https://app.posthog.com)
 *   - POSTHOG_PERSONAL_API_KEY (for management scripts, not bundled with the app)
 *   - POSTHOG_ANALYTICS_DEBUG_ENABLED (truthy string to enable writing analytics decisions to a local log file)
 *   - POSTHOG_ANALYTICS_DEBUG_RETAIN_AFTER_SEND (truthy string to keep log entries even after events are sent)
 *   - POSTHOG_ANALYTICS_DEBUG_LOG_FILE (relative path for the analytics log file; defaults to analytics-events.log)
 *
 * Usage:
 *   POSTHOG_PROJECT_API_KEY=phc_123 npm run setup:posthog
 */

const fs = require('fs');
const path = require('path');

const projectApiKey = process.env.POSTHOG_PROJECT_API_KEY;
const apiHost = process.env.POSTHOG_API_HOST || 'https://app.posthog.com';
const personalApiKey = process.env.POSTHOG_PERSONAL_API_KEY || null;
function parseBoolean(value) {
  if (value == null) {
    return false;
  }
  if (typeof value === 'boolean') {
    return value;
  }
  const normalized = String(value).toLowerCase();
  return normalized !== '0' && normalized !== 'false' && normalized !== 'no' && normalized !== '';
}

const analyticsDebugEnabled = parseBoolean(process.env.POSTHOG_ANALYTICS_DEBUG_ENABLED);
const analyticsDebugRetain = parseBoolean(process.env.POSTHOG_ANALYTICS_DEBUG_RETAIN_AFTER_SEND);
const analyticsDebugLogFile = process.env.POSTHOG_ANALYTICS_DEBUG_LOG_FILE || 'analytics-events.log';

if (!projectApiKey) {
  console.error(
    '[posthog] Missing required POSTHOG_PROJECT_API_KEY. See docs/analytics/posthog.md for setup instructions.'
  );
  process.exitCode = 1;
  return;
}

const configDir = path.join(__dirname, '..', 'config');
const outputPath = path.join(configDir, 'posthog.json');

fs.mkdirSync(configDir, { recursive: true });

const config = {
  projectApiKey,
  apiHost,
  personalApiKey,
  featureFlags: {
    pollingIntervalMs: 30000,
    // Set to true only if feature flags must be evaluated before renderer loads.
    preloadOnLaunch: false
  },
  capture: {
    autocapture: false,
    sessionRecording: false
  },
  analyticsDebug: {
    enabled: analyticsDebugEnabled,
    retainAfterSend: analyticsDebugRetain,
    logFile: analyticsDebugLogFile
  }
};

fs.writeFileSync(outputPath, `${JSON.stringify(config, null, 2)}\n`, 'utf8');

console.info(`[posthog] Wrote credentials to ${path.relative(process.cwd(), outputPath)}`);
if (personalApiKey) {
  console.info('[posthog] Personal API key configured for tooling.');
}

const eventsConfigUrl = process.env.POSTHOG_EVENTS_CONFIG_URL || null;
if (eventsConfigUrl) {
  const eventsConfigPath = path.join(configDir, 'posthog-events.json');
  let eventsConfig = null;

  if (fs.existsSync(eventsConfigPath)) {
    try {
      eventsConfig = JSON.parse(fs.readFileSync(eventsConfigPath, 'utf8'));
    } catch (error) {
      console.warn(
        '[posthog] Failed to parse existing posthog-events.json, generating a new template.'
      );
    }
  }

  if (!eventsConfig || typeof eventsConfig !== 'object') {
    eventsConfig = {
      version: 1,
      remote: {
        url: eventsConfigUrl,
        etag: null,
        timeoutMs: 5000
      },
      defaults: {
        allow: false,
        propertyAllowlist: ['source', 'registry_version', 'analytics_debug_mode'],
        propertyTransforms: {},
        dropReasons: []
      },
      events: {}
    };
  } else {
    const remote = eventsConfig.remote && typeof eventsConfig.remote === 'object' ? eventsConfig.remote : {};
    eventsConfig.remote = {
      ...remote,
      url: eventsConfigUrl,
      timeoutMs: typeof remote.timeoutMs === 'number' ? remote.timeoutMs : 5000,
      etag: null
    };
  }

  fs.mkdirSync(configDir, { recursive: true });
  fs.writeFileSync(eventsConfigPath, `${JSON.stringify(eventsConfig, null, 2)}\n`, 'utf8');
  console.info(`[posthog] Updated events config remote URL to ${eventsConfigUrl}`);
}
